package mcfall.raytracer.objects;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import mcfall.math.LineSegment;
import mcfall.math.Matrix;
import mcfall.math.Point;
import mcfall.math.Ray;
import mcfall.raytracer.Camera;
import mcfall.raytracer.Material;

/**
 * The Class GenericBuckyBall.  Creates a bucky ball centered at (0,0,0) with a maximal radius of Math.sqrt(9d*phi+10d)+1d.  This is cononical since the edge lengths are 2 (which is traditional with this shape)
 */
public class GenericBuckyBall extends  AbstractBoundingBox {
	
	/** The Constant phi. */
	private static final double phi = (1d+Math.sqrt(5d))/2d;
	
	/** The Constant radius. */
	private static final double radius = Math.sqrt(9d*phi+10d)+1d;
	
	/** The Constant bbDiagonal which repressents the diagonal of the bounding box. */
	private static final double bbDiagonal = Math.sqrt(Math.pow(radius+1d,2d)+Math.pow((radius+1d)/2d, 2d));   
	
	/** The vertecies. */
	private static Set<Point> vertecies = new HashSet<Point>(60);
	
	/** The edges. */
	private static List<LineSegment> edges = new LinkedList<LineSegment>();
	
	/** The spheres at the edges. */
	private Set<GenericSphere> spheres=null;
	/** The static constructor */
	{
		createVertecies();
		createEdges();
	}
	
	/**
	 * Instantiates a new generic bucky ball.
	 */
	public GenericBuckyBall() {
		super(new Point(0,0,0),new Point(bbDiagonal,bbDiagonal,bbDiagonal));
		createSpheres();
		
	}
	
	/**
	 * Instantiates a new generic bucky ball.
	 * 
	 * @param name the name
	 */
	public GenericBuckyBall(String name){
		super(name, new Point(0,0,0),new Point(bbDiagonal,bbDiagonal,bbDiagonal));
		createSpheres();
	}	
	
	/**
	 * Permute.
	 * 
	 * @param p the p
	 * 
	 * @return the set< point>
	 */
	private static Set<Point> permute(Point p) {
		HashSet<Point> points = new HashSet<Point>();
		for(int x=-1;x==-1 || (x<=1 && p.getX()!=0);x+=2) {//ensure that we don't put 2 points of x with the same coordinant (-1*0 and 1*0) since sets DO NOT do .equals they do a hashCode comparison to ensure uniquness
			for(int y=-1;y==-1 || (y<=1 && p.getY()!=0);y+=2) {
				for(int z=-1;z==-1 || (z<=1 && p.getZ()!=0);z+=2) {
					if(!points.contains(p)) {
						points.add(new Point(p.getX()*x,p.getY()*y,p.getZ()*z));
					}
				}
			}
		}
		return points;
	}
	
	/**
	 * Creates the vertecies.
	 */
	private static void createVertecies() {
		if(vertecies.size()==0) {
			vertecies.addAll(permute(new Point(0,1d,3d*phi)));
			vertecies.addAll(permute(new Point(1d,3d*phi,0)));
			vertecies.addAll(permute(new Point(3d*phi,0,1d)));
			
			vertecies.addAll(permute( new Point(2d,1d+2d*phi,phi) ));
			vertecies.addAll(permute( new Point(1d+2d*phi,phi,2d) ));
			vertecies.addAll(permute( new Point(phi,2d,1d+2d*phi) ));
			
			vertecies.addAll(permute( new Point(1d,2d+phi,2d*phi) ));
			vertecies.addAll(permute( new Point(2d+phi,2d*phi,1d) ));
			vertecies.addAll(permute( new Point(2d*phi,1d,2d+phi) ));
			System.out.println(vertecies.size());
		}
	}
	
	/**
	 * Creates the edges.
	 */
	private static void createEdges() {
		if(edges.size()==0) {
			createVertecies();
			for(Point vertex1: vertecies) {
				for(Point vertex2: vertecies) {
					LineSegment edge = new LineSegment(vertex1,vertex2);
					if(edge.length()==2 && !edges.contains(edge)) { //all of the nearest edges have length 2 and dont put any in that we dont need (since we cant use a set)
						edges.add(edge);
					}
				}
			}
			System.out.println(edges.size());
		}
	}
	
	/**
	 * Creates the spheres.
	 */
	private void createSpheres() {
		if(spheres==null) {
			spheres = new HashSet<GenericSphere>();
			for(Point v : GenericBuckyBall.vertecies) {
				GenericSphere s = new GenericSphere();
				s.transform(Matrix.createTranslationMatrix(v.getX()*2d, v.getY()*2d, v.getZ()*2d));
				spheres.add(s);
			}
			System.out.println(spheres.size());
		}
	}
	
	/* (non-Javadoc)
	 * @see mcfall.raytracer.objects.MathematicalObject#genericHitTime(mcfall.math.Ray)
	 */
	@Override
	protected List<HitRecord> genericHitTime(Ray ray) {
		TreeSet<HitRecord> hr = new TreeSet<HitRecord>();
		for(GenericSphere s : spheres) {
			List<HitRecord> hits = s.hitTime(ray);
			if(hits.size()>0) {
				hr.addAll(hits);
			}
		}
		return new LinkedList<HitRecord>(hr);
	}

	/* (non-Javadoc)
	 * @see mcfall.raytracer.objects.MathematicalObject#getObjectType()
	 */
	@Override
	protected String getObjectType() {
		return "Generic Bucky Ball";
	}
	
	/* (non-Javadoc)
	 * @see mcfall.raytracer.objects.AbstractBoundingBox#transform(mcfall.math.Matrix)
	 */
	@Override
	public void transform(Matrix transform) {
		super.transform(transform);
		for(GenericSphere s : spheres) {
			s.transform(transform);
		}
	}
	
	/* (non-Javadoc)
	 * @see mcfall.raytracer.objects.MathematicalObject#setMaterial(mcfall.raytracer.Material)
	 */
	@Override
	public void setMaterial(Material material) {
		super.setMaterial(material);
		for(GenericSphere s : spheres) {
			s.setMaterial(material);
		}
	}
	
	/* (non-Javadoc)
	 * @see mcfall.raytracer.objects.MathematicalObject#setReflectionCoefficient(double)
	 */
	@Override
	public void setReflectionCoefficient(double reflectivity) {
		super.setReflectionCoefficient(reflectivity);
		for(GenericSphere s : spheres) {
			s.setReflectionCoefficient(reflectivity);
		}
	}
	
	/* (non-Javadoc)
	 * @see mcfall.raytracer.objects.MathematicalObject#setRefractionIndex(double)
	 */
	@Override
	public void setRefractionIndex(double refractionPercent) {
		super.setRefractionIndex(refractionPercent);
		for(GenericSphere s : spheres) {
			s.setRefractionIndex(refractionPercent);
		}
	}
	
	/* (non-Javadoc)
	 * @see mcfall.raytracer.objects.MathematicalObject#setTransparency(double)
	 */
	@Override
	public void setTransparency(double transparency) {
		super.setTransparency(transparency);
		for(GenericSphere s : spheres) {
			s.setTransparency(transparency);
		}
	}
	
	
	

}
